#!/bin/bash

set -euo pipefail

DIR="$(cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd)"
. "$DIR/shared.sh"

echo "Updating packages"
apt-update
apt-upgrade

echo "Adding user ubuntu"
user_add_sudo ubuntu
usermod -s /usr/bin/zsh ubuntu
cp /root/.zshrc /home/ubuntu/.zshrc
chown ubuntu:ubuntu /home/ubuntu/.zshrc

echo "Hardening ssh"
apt-install openssh-server
cat > /etc/ssh/sshd_config <<EOM
AuthorizedKeysFile .ssh/authorized_keys
ClientAliveInterval 180
Subsystem sftp /usr/lib/openssh/sftp-server
UseDNS no
PermitRootLogin no
UsePAM yes

KexAlgorithms curve25519-sha256@libssh.org
Protocol 2
HostKey /etc/ssh/ssh_host_ed25519_key
PasswordAuthentication no
ChallengeResponseAuthentication no
PubkeyAuthentication yes
Ciphers chacha20-poly1305@openssh.com,aes256-gcm@openssh.com,aes128-gcm@openssh.com,aes256-ctr,aes192-ctr,aes128-ctr
MACs hmac-sha2-512-etm@openssh.com,hmac-sha2-256-etm@openssh.com,umac-128-etm@openssh.com,hmac-sha2-512,hmac-sha2-256,umac-128@openssh.com
EOM

cat > /etc/ssh/ssh_config <<EOM
Host *
  KexAlgorithms curve25519-sha256@libssh.org,diffie-hellman-group-exchange-sha256
  PasswordAuthentication no
  ChallengeResponseAuthentication no
  PubkeyAuthentication yes
  HostKeyAlgorithms ssh-ed25519-cert-v01@openssh.com,ssh-rsa-cert-v01@openssh.com,ssh-ed25519,ssh-rsa
  Ciphers chacha20-poly1305@openssh.com,aes256-gcm@openssh.com,aes128-gcm@openssh.com,aes256-ctr,aes192-ctr,aes128-ctr
  MACs hmac-sha2-512-etm@openssh.com,hmac-sha2-256-etm@openssh.com,umac-128-etm@openssh.com,hmac-sha2-512,hmac-sha2-256,umac-128@openssh.com
  UseRoaming no
EOM
systemctl restart ssh

echo "Setting up firewall"
mkdir -p /etc/iptables
cat > /etc/iptables/rules.v4 <<EOM
*nat
:PREROUTING ACCEPT [0:0]
:INPUT ACCEPT [0:0]
:OUTPUT ACCEPT [0:0]
:POSTROUTING ACCEPT [0:0]
COMMIT
*filter
:INPUT ACCEPT [0:0]
:FORWARD DROP [0:0]
:OUTPUT ACCEPT [0:0]
:DOCKER-USER - [0:0]
:icmp-routing -
:logdrop-0 -

-A DOCKER-USER -i eth0 -m conntrack --ctstate NEW -j logdrop-0
-A DOCKER-USER -j RETURN

-A INPUT -m conntrack --ctstate RELATED,ESTABLISHED -j ACCEPT
-A INPUT -i lo -m comment --comment "Allow loopback connections" -j ACCEPT
-A INPUT -p icmp -m comment --comment "Allow Ping to work as expected" -j ACCEPT
-A INPUT -i eth0 -p tcp --dport 22 -j ACCEPT
-A INPUT -p udp --sport 67 --dport 68 -j ACCEPT
-A INPUT -p udp -m multiport --dports 60000:61000 -j ACCEPT
-A INPUT -i eth0 -j logdrop-0

-A FORWARD -i eth0 -j logdrop-0

-A logdrop-0 -m limit --limit 1/second -j LOG
-A logdrop-0 -j DROP
COMMIT
EOM

cat > /etc/iptables/rules.v6 <<EOM
*filter
:INPUT ACCEPT [0:0]
:FORWARD ACCEPT [0:0]
:OUTPUT ACCEPT [0:0]

# Allow all loopback (lo0) traffic and reject traffic
# to localhost that does not originate from lo0.
-A INPUT -i lo -j ACCEPT
-A INPUT ! -i lo -s ::1/128 -j REJECT

# Below are the rules which are required for your IPv6 address to be properly allocated
-A INPUT -p icmpv6 --icmpv6-type router-advertisement -m hl --hl-eq 255 -j ACCEPT
-A INPUT -p icmpv6 --icmpv6-type neighbor-solicitation -m hl --hl-eq 255 -j ACCEPT
-A INPUT -p icmpv6 --icmpv6-type neighbor-advertisement -m hl --hl-eq 255 -j ACCEPT
-A INPUT -p icmpv6 --icmpv6-type redirect -m hl --hl-eq 255 -j ACCEPT

# Allow ICMP
-A INPUT -p icmpv6 -j ACCEPT

# Allow inbound traffic from established connections.
-A INPUT -m state --state ESTABLISHED,RELATED -j ACCEPT

# Log what was incoming but denied (optional but useful).
-A INPUT -m limit --limit 5/min -j LOG --log-prefix "ip6tables_INPUT_denied: " --log-level 7

# Reject all other inbound.
-A INPUT -j REJECT

# Log any traffic that was sent to you
# for forwarding (optional but useful).
-A FORWARD -m limit --limit 5/min -j LOG --log-prefix "ip6tables_FORWARD_denied: " --log-level 7

# Reject all traffic forwarding.
-A FORWARD -j REJECT

COMMIT
EOM
save_firewall

echo "Installing docker"
apt-install \
    apt-transport-https \
    ca-certificates \
    curl \
    gnupg-agent \
    software-properties-common

curl -fsSL https://download.docker.com/linux/ubuntu/gpg | gpg --dearmor -o /usr/share/keyrings/docker-archive-keyring.gpg
echo \
  "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker-archive-keyring.gpg] https://download.docker.com/linux/ubuntu \
  $(lsb_release -cs) stable" | tee /etc/apt/sources.list.d/docker.list > /dev/null

apt-update
apt-install docker-ce docker-ce-cli containerd.io docker-buildx-plugin docker-compose-plugin

groupadd docker || true
usermod -aG docker ubuntu

systemctl enable docker

echo "Installing packages"
apt-install unzip amazon-ecr-credential-helper gosu build-essential zsh fontconfig python3-pip libnss3-tools

echo "Installing Homebrew"
user-mkdir /home/linuxbrew/.linuxbrew
gosu ubuntu /bin/bash -c "NONINTERACTIVE=1 USER=ubuntu $(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"
echo 'eval "$(/home/linuxbrew/.linuxbrew/bin/brew shellenv)"' >> /home/ubuntu/.bashrc
echo 'eval "$(/home/linuxbrew/.linuxbrew/bin/brew shellenv)"' >> /home/ubuntu/.zshrc
chown ubuntu:ubuntu /home/ubuntu/.bashrc
chown ubuntu:ubuntu /home/ubuntu/.zshrc

echo "Installing github cli"
user-brew install gh

echo "Installing mkcert"
user-brew install mkcert

pushd /home/ubuntu

echo "Installing lastversion"
gosu ubuntu /bin/bash -c "USER=ubuntu pip3 install lastversion"

popd

echo "Installing ansible"
pip3 install ansible

echo "Enable user-ssh-key firstboot script"
systemctl enable user-ssh-key

echo "Enable multiarch service"
systemctl enable multiarch

echo "Installing aws cli"
curl -fsSL "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"
unzip awscliv2.zip
./aws/install
rm awscliv2.zip
rm -rf ./aws

echo "Installing brew packages"
user-brew install bat starship hub git-delta fzf ripgrep jq neovim git-lfs fd exa

echo "Installing et"
add-apt-repository -y ppa:jgmath2000/et
apt-update
apt-install et
systemctl enable et

user-mkdir /home/ubuntu/.docker
cat << EOF > /home/ubuntu/.docker/config.json
{
  "credHelpers": {
    "786715713882.dkr.ecr.us-east-1.amazonaws.com" : "ecr-login",
    "public.ecr.aws": "ecr-login"
  }
}
EOF

echo "Making zsh the default shell"
sudo sed -i 's/DSHELL=\/bin\/bash/DSHELL=\/usr\/bin\/zsh/' /etc/adduser.conf

echo "Provisioning complete!"
